#!/bin/bash

# Function to print help message
print_help() {
  echo "Description:"
  echo "  This script traverses a directory and prints the contents of files in markdown format."
  echo "  It was designed to simplify sharing code from a project with ChatGPT by formatting the files as markdown text."
  echo "  By default, it only includes files of type 'application' or 'text'."
  echo ""
  echo "  Example output:"
  echo ""
  echo "    ./example.txt"
  echo "    "'```'
  echo "    This is an example file content."
  echo "    "'```'
  echo ""
  echo ""
  echo "Usage: printFilesAsMdCode [directory] [options]"
  echo ""
  echo "Options:"
  echo "  --clipboard, -c            Copy output to clipboard"
  echo "  --ignore-types, -i         Ignore file types and output all files"
  echo "  --include-hidden-files, -f Include hidden files"
  echo "  --include-hidden-dirs, -d  Include hidden directories"
  echo "  --help, -h                 Display this help message"
  echo ""
  echo "Examples:"
  echo "  printFilesAsMdCode"
  echo "    # Outputs files from the current directory."
  echo "  printFilesAsMdCode /path/to/directory"
  echo "    # Outputs files from the specified directory."
  echo "  printFilesAsMdCode /path/to/directory --clipboard --ignore-types --include-hidden-files --include-hidden-dirs"
  echo "  printFilesAsMdCode /path/to/directory -c -i -f -d"
  echo "    # Outputs all files, from the specified directory including hidden files and directories, to the clipboard."
}

# Function to output file contents
output_file_content() {
  local filepath=$1
  echo ""
  echo "$filepath"
  echo '```'
  cat "$filepath"
  echo '```'
  echo ""
}

# Default settings
DIR=$(pwd)
CLIPBOARD=false
IGNORE_TYPES=false
INCLUDE_HIDDEN_FILES=false
INCLUDE_HIDDEN_DIRS=false

# Parse flags and arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --clipboard|-c)
      CLIPBOARD=true
      shift
      ;;
    --ignore-types|-i)
      IGNORE_TYPES=true
      shift
      ;;
    --include-hidden-files|-f)
      INCLUDE_HIDDEN_FILES=true
      shift
      ;;
    --include-hidden-dirs|-d)
      INCLUDE_HIDDEN_DIRS=true
      shift
      ;;
    --help|-h)
      print_help
      exit 0
      ;;
    *)
      DIR=$1
      shift
      ;;
  esac
done

# Create a temporary file for clipboard
TEMP_FILE=$(mktemp)

# Build find command based on hidden file/dir inclusion
FIND_CMD="find \"$DIR\" -type f"
if ! $INCLUDE_HIDDEN_FILES; then
  FIND_CMD="$FIND_CMD ! -name '.*'"
fi
if ! $INCLUDE_HIDDEN_DIRS; then
  FIND_CMD="$FIND_CMD ! -path '*/.*/*'"
fi

# Traverse directory and output contents
{
  eval "$FIND_CMD" | while read -r file; do
    if $IGNORE_TYPES; then
      relative_path="${file#"$DIR"/}"
      output_file_content "$relative_path"
    else
      file_type=$(xdg-mime query filetype "$file")
      major_type=$(echo "$file_type" | cut -d'/' -f1)
      if [[ "$major_type" == "application" || "$major_type" == "text" ]]; then
        relative_path="${file#"$DIR"/}"
        output_file_content "$relative_path"
      fi
    fi
  done
} | tee "$TEMP_FILE"

# Copy to clipboard if flag is set
if $CLIPBOARD; then
  if command -v wl-copy &> /dev/null; then
    wl-copy < "$TEMP_FILE"
  elif command -v xclip &> /dev/null; then
    xclip -selection clipboard < "$TEMP_FILE"
  else
    echo "No clipboard utility found. Please install wl-clipboard or xclip."
  fi
  echo "Output copied to clipboard."
fi

# Cleanup
rm "$TEMP_FILE"
