#!/usr/bin/python
# -*- coding: utf-8 -*-
import sys
import re
import argparse

def extract_colors(css_file, output_file):
    color_pattern = re.compile(r'(#(?:[0-9a-fA-F]{3}){1,2})|'
                               r'(rgba?\(\s*\d+\s*,\s*\d+\s*,\s*\d+\s*(,\s*\d*\.?\d+\s*)?\))')
    colors = set()

    with open(css_file, 'r') as file:
        for line in file:
            matches = color_pattern.finditer(line)
            for match in matches:
                color = match.group(0)
                colors.add(color)

    with open(output_file, 'w') as file:
        for color in colors:
            file.write(color + '\n')

def replace_colors(css_file, translation_file, output_file):
    with open(translation_file, 'r') as file:
        translations = dict(line.strip().split(' -> ') for line in file)

    with open(css_file, 'r') as file:
        css_content = file.read()

    for old_color, new_color in translations.items():
        css_content = css_content.replace(old_color, new_color)

    with open(output_file, 'w') as file:
        file.write(css_content)

def main():
    parser = argparse.ArgumentParser(description="CSS Color Extractor and Replacer")
    parser.add_argument("command", choices=['extract', 'replace'], help="Command to execute (extract or replace)")
    parser.add_argument("css_file", help="Path to the CSS file")
    parser.add_argument("output_file", help="Path to the output file")
    parser.add_argument("--translation_file", help="Path to the translation file for color replacement", default="")

    args = parser.parse_args()

    if args.command == 'extract':
        extract_colors(args.css_file, args.output_file)
    elif args.command == 'replace':
        if not args.translation_file:
            print("Error: Translation file required for replace command")
            return
        replace_colors(args.css_file, args.translation_file, args.output_file)

if __name__ == "__main__":
    main()
