#!/usr/bin/env bash

# User-configurable variables
containerName="mongo-db-name"
dbUser="mongo-user"
dbPassword="mongo-password"
dbName="mongo-db"

hostPort=27017
containerPort=27017

initScriptPath="/tmp/init-mongo.js"


# Function to create the init script for mongoDB
create_init_script() {
  echo "Creating MongoDB initialization script at $initScriptPath..."
  cat <<EOF >"$initScriptPath"
db.createUser({
    user: '$dbUser',
    pwd: '$dbPassword',
    roles: [{
        role: 'readWrite',
        db: '$dbName'
    }]
});
EOF
}


# Function to create a new container
create_container() {
  echo "Creating container '$containerName'..."
  create_init_script
  docker run \
    --name "$containerName" \
    -e MONGO_INITDB_DATABASE="$dbName" \
    -v "$initScriptPath:/docker-entrypoint-initdb.d/init-mongo.js" \
    -p "$hostPort:$containerPort" \
    -d mongo
  # Remove the init script after starting the container
  rm "$initScriptPath"
}


# Function to start an existing container
start_container() {
  echo "Starting container '$containerName'..."
  docker start "$containerName"
}


# Main script logic

# Check whether the container exists (in any state)
containerExists=$(docker ps -a --filter "name=$containerName" --format "{{.Names}}")

if [ -n "$containerExists" ]; then
  # Container exists, check if it's running
  containerRunning=$(docker ps --filter "name=$containerName" --filter "status=running" --format "{{.Names}}")
  if [ -n "$containerRunning" ]; then
    echo "Container '$containerName' is already running."
  else
    echo "Container '$containerName' exists but is not running."
    if ! start_container; then
      echo "Failed to start container. Removing and recreating it..."
      docker rm "$containerName"
      create_container
    fi
  fi
else
  # Container does not exist, create it
  create_container
fi
