#!/usr/bin/env bash
set -euo pipefail

# defaults
sort="name"
depth=1
dirs=()

# Function to print help message
print_help() {
  echo "Description:"
  echo "  Displays a directory tree with human-readable sizes."
  echo "  Under the hood it uses GNU tree with: -s --du -L --sort ."
  echo ""
  echo ""
  echo "Usage: filesize [directory...] [options]"
  echo ""
  echo "Options:"
  echo "  -h, --help             Show this help message and exit"
  echo "  -s, --sort SORT        Sort by: name, version, size, mtime, ctime, none"
  echo "  -d, --depth N          Descend only N levels deep"
  echo ""
  echo ""
  echo "Examples:"
  echo "  # Show current directory, name-sorted"
  echo "  filesize"
  echo ""
  echo "  # Three levels deep, size-sorted"
  echo "  filesize --depth 3 --sort size"
  echo "  filesize -d 3 -s size"
  echo ""
  echo "  # On /var/log, two levels deep, by modification time"
  echo "  filesize /var/log --depth 2 --sort mtime"
  echo "  filesize /var/log -d 2 -s mtime"
  echo ""
  exit 0
}

# parse flags
while [[ $# -gt 0 ]]; do
  case $1 in
    -h|--help)
      print_help
      ;;
    -s|--sort)
      if [[ -z "${2-}" ]]; then
        echo "Error: --sort requires an argument" >&2
        exit 1
      fi
      sort=$2
      shift
      ;;
    -d|--depth)
      if [[ -z "${2-}" ]]; then
        echo "Error: --depth requires an argument" >&2
        exit 1
      fi
      depth=$2
      shift
      ;;
    --*)
      echo "Unknown option: $1" >&2
      exit 1
      ;;
    -*)
      echo "Unknown short option: $1" >&2
      exit 1
      ;;
    *)
      dirs+=("$1")
      ;;
  esac
  shift
done

# if no dirs specified, use current
if [[ ${#dirs[@]} -eq 0 ]]; then
  dirs=(.)
fi

# run
exec tree -sh --du -L "$depth" --sort "$sort" "${dirs[@]}"
